/* TRABI: Temporal, Radial-Aperture Based Intensity-estimation
 *  
 *  Requires ImageJ 1.51k or higher.
 *  
 *  Determines intensity of fluorescent spots in a TIF-stack from a localization file 
 *  (rapidSTORM, ThunderSTORM, PeakFit etc). For each coordinate, a circular selection 
 *  is made and the integrated density (I_raw) is determined. The corrected intensity 
 *  (I_1) is calculated by subtracting background signal (I_BG) from I_raw. I_BG is averaged 
 *  over subsequent frames. ROIs on edges are discarded.
 *  
 *  License: This version of TRABI (1.1) can be used under the terms of GNU GPLv3.
 *  
 *  2015-2017
 *  Author: Sebastian van de Linde
 *    Department of Physics, University of Strathclyde, Glasgow, Scotland
 *  Email: s.vandelinde@strath.ac.uk
 */

/******************************* DIALOG *******************************/
version = "1.1";

presets_file = getDirectory("macros") + "TRABI_v1.1_settings.txt";
presets_exist = File.exists(presets_file);
preset = loadTxtFile(presets_file, 1, presets_exist);
items = newArray("rapidSTORM", "ThunderSTORM", "PeakFit", "TRABI_output", "Basic");
items_output = newArray("Basic", "Basic+FWHM", "Basic+BG", "Complete");
html = html_message();

Dialog.create("TRABI " + version);
Dialog.setInsets(0,20,0);
Dialog.addCheckbox("BATCH PROCESSING", preset[0]);
Dialog.setInsets(0,0,0);
Dialog.addMessage("I. Input options");
Dialog.addChoice("Loc-file", items, preset[1]);
Dialog.setInsets(0,20,0);
Dialog.addCheckbox("Free PSF fit", preset[2]);
Dialog.addCheckbox("Coordinates in nm", preset[3]);
Dialog.addNumber("Pixelsize", preset[4], 0, 4, "nm");
Dialog.setInsets(0,0,0);
Dialog.addMessage("II. Output options");
Dialog.addChoice("Output file", items_output, preset[5]);
Dialog.addCheckbox("P-filter activated", preset[6]);
Dialog.addNumber("P-min ", preset[7], 1, 5, "%");
Dialog.addNumber("P-max ", preset[8], 1, 5, "%");
Dialog.addCheckbox("BG-Averaging: save frame info", preset[9]); 
Dialog.addCheckbox("Save settings to file name", preset[10]); 
Dialog.setInsets(0,0,0);
Dialog.addMessage("III. TRABI settings");
Dialog.addNumber("r1", preset[11], 0, 3, ".5 px");
Dialog.addNumber("Basejump", preset[12], 0, 3, "frame(s)"); 
Dialog.addNumber("nBG", preset[13], 0, 3, "frame(s)");
Dialog.addChoice("I2 by", newArray("Fit_only", "Fit+Circle"), preset[14]);
Dialog.addNumber("EZ factor", preset[15], 1, 2,""); 
Dialog.addCheckbox("Highlander filter (HF)", preset[16]);
Dialog.addNumber("HF", preset[17], 0, 4, "frame(s)"); 
Dialog.addHelp(html);
Dialog.show();
batch_processing = Dialog.getCheckbox();
input_loc_file = Dialog.getChoice();
free_fit = Dialog.getCheckbox();
pixel_size_in_nm = Dialog.getCheckbox();
pixelsize = Dialog.getNumber();
output_file_option = Dialog.getChoice();
threshold_P_value = Dialog.getCheckbox();
lower_P_threshold = Dialog.getNumber();
upper_P_threshold = Dialog.getNumber();
background_info = Dialog.getCheckbox();
file_name_settings = Dialog.getCheckbox();
radius = Dialog.getNumber(); 
basejump = Dialog.getNumber();
average_bg_number = Dialog.getNumber();
I2_determination = Dialog.getChoice();
exclusion_zone_factor = Dialog.getNumber();
exclude_highlander = Dialog.getCheckbox();
highlander_threshold = Dialog.getNumber();

if(I2_determination == "Fit+Circle"){
	Dialog.create("TRABI " + version);
	Dialog.addMessage("Circular aperture radius");
	Dialog.addNumber("r2", preset[18], 0, 0, ".5 px");
	Dialog.show;
	radius2 = Dialog.getNumber();
}
if(input_loc_file == "TRABI_output"){
	Dialog.create("TRABI " + version);
	Dialog.addMessage("Specify start frame");
	Dialog.addChoice("", newArray(0, 1), 0);
	Dialog.show;
	frame_start_number = Dialog.getChoice();
}
if(batch_processing == true){
	Dialog.create("Batch processing");
	Dialog.addMessage("Specify file extensions");
	Dialog.addString(" Image file extension", preset[19], 2);
	Dialog.addString(" Loc-file extension", preset[20], 2); 
	Dialog.show;
	image_file_ending = Dialog.getString();
	loc_file_ending = Dialog.getString();
}
// Batch mode
setBatchMode(true);

/******************************* SETTINGS *******************************/

if(input_loc_file == "ThunderSTORM") pixel_size_in_nm = true;
else if(input_loc_file == "rapidSTORM") pixel_size_in_nm = true;
if(pixel_size_in_nm == false) pixelsize = 1;
if(threshold_P_value == false){lower_P_threshold = NaN; upper_P_threshold = NaN;}
radius_increment = 0.5; 
radius_nm = radius*pixelsize;
radius_increment_nm = radius_increment*pixelsize;
exclusion_zone_radius_nm = exclusion_zone_factor*(radius_nm + radius_increment_nm);
spacer = " ";

//TRABI FILE NAME
if(file_name_settings == true){
	suffix = newArray("TRABI", "v"+version,"px-size",pixelsize,"r1",d2s(radius, 0)+".5px","bj",basejump,"nBG",average_bg_number,"EZ-factor",exclusion_zone_factor);
	if(I2_determination == "Fit+Circle"){
		suffix_r2 = newArray("r2",d2s(radius2, 0)+".5px");
		suffix = Array.concat(suffix, suffix_r2);
	}
	if(exclude_highlander == true){
		suffix_hf = newArray("HF", highlander_threshold);
		suffix = Array.concat(suffix, suffix_hf);
	}
} else suffix = newArray("TRABI", "v"+version);
suffix = print2line(suffix, "_");

if(input_loc_file == "rapidSTORM") header = newArray(".*Position-0-0.*", "x", ".*Position-1-0.*", "y", ".*ImageNumber-0-0.*", "frame", ".*Amplitude-0-0.*", "ADC", ".*TwoKernelImprovement-0-0.*", "TwoKernel", ".*FitResidues-0-0.*", "FitResidues", ".*LocalBackground-0-0.*", "LocalBackground", ".*Position-2-0.*", "z", ".*PSFWidth-0-0.*", "PSFwidthFWHMx", ".*PSFWidth-1-0.*", "PSFwidthFWHMy");
else if(input_loc_file == "ThunderSTORM") header = newArray("\"x [nm]\"","\"y [nm]\"", "\"frame\"","\"intensity [photon]\"","\"sigma [nm]\"","\"offset [photon]\"","\"bkgstd [photon]\"","\"chi2\"","\"uncertainty [nm]\"");
else if(input_loc_file == "PeakFit") header = newArray("X","Y", "Frame","Signal", "SNR", "X SD", "Y SD");
else if(input_loc_file == "TRABI_output") header = newArray("#X", "Y", "Frame", "I_1");
else if(input_loc_file == "Basic") header = newArray("#X", "Y", "Frame", "Intensity");

if(input_loc_file == "TRABI_output") first_line_with_coordinates = 2;
else first_line_with_coordinates = 1;

if(input_loc_file == "ThunderSTORM") delimiter_h = ",";
else if(input_loc_file == "PeakFit") delimiter_h = "\t";
else delimiter_h = " ";

/****************************** START ************************************/

print("\\Clear");

// Batch processing
if(batch_processing == true){
	dir1 = getDirectory("Select directory with files ");
	dir2 = dir1 + "TRABI" + File.separator;
	File.makeDirectory(dir2);
	list = getFileList(dir1);
	list_image_file = newArray();
	list_loc_file = newArray();
	for(a=0; a<list.length; a++){
		filename = dir1 + list[a];
		if(indexOf(filename, image_file_ending)>0) list_image_file = Array.concat(list_image_file, filename);
		if(indexOf(filename, loc_file_ending)>0) list_loc_file = Array.concat (list_loc_file, filename);
	}
	if(list_image_file.length != list_loc_file.length) exit("Uneven ratio of stacks to loc-files");// ERROR MESSAGE
} else { // single file
	list_image_file = newArray("");
	list_image_file[0] = getTitle();
}

for(ff=0; ff<list_image_file.length; ff++){
	imagename = list_image_file[ff];
	if(batch_processing == true){
		filename = list_loc_file[ff];
		open(imagename);
		imagename_short = filenameDistillery(imagename, File.directory);
	} else {
		filename = File.openDialog("Select a localization file");
		imagename_short = imagename;
	}
	filename_short = filenameDistillery(filename, File.directory);
	print("Fileset", ff+1);
	print("Loc-file: " + filename_short);
	print("Stack: " + imagename_short);
	time1 = getTime;
	if(input_loc_file == "PeakFit" && batch_processing == true) header = newArray("X","Y", "Frame","Signal", "SNR", "X SD", "Y SD");
	loadLocFile(filename, header, first_line_with_coordinates, delimiter_h);
	if(endsWith(filename, ".csv") == true) file_ending = ".csv";
	else file_ending = ".txt";
	if(batch_processing == true){
		filename_short = suffixEliminator(filename_short, file_ending);
		general_filename = dir2 + filename_short + "_" + suffix;
	}
	else {
		filename = suffixEliminator(filename, file_ending);
		general_filename = filename + "_" + suffix;
	}
	// TRABI output file
	TRABI_output_file = File.open(general_filename + ".txt");
	
	// Header
	header_intensity_file = printHeader(output_file_option, input_loc_file, free_fit);
	print2file(TRABI_output_file, header_intensity_file, spacer);
	header_add_info = newArray("#Radius1:",radius,"+",radius_increment,"px"," Pixelsize:",pixelsize,"nm"," basejump:",basejump," n_BG:",average_bg_number," P-low-Threshold:",lower_P_threshold," P-up-Threshold:",upper_P_threshold," HF-Threshold:" , highlander_threshold," EZ_factor:" , exclusion_zone_factor);
	if(I2_determination == "Fit+Circle"){
		radius2_info = " Radius2:"+radius2+".5px";
		header_add_info = Array.concat(header_add_info, radius2_info);
	}
	header_add_info = print2line(header_add_info, "");
	print(TRABI_output_file, header_add_info);
	
	//Image properties
	run("Properties...", "pixel_width=1 pixel_height=1 voxel_depth=1");
	x_max = getWidth();
	y_max = getHeight();
	if(input_loc_file == "rapidSTORM"){
		x_header = header[1];
		y_header = header[3];
		frame_header = header[5];
		intensity_header = header[7];
		rs_px_shift = 0.5 * pixelsize;
		for(i=0; i<nResults; i++){
			x_coor_raw = getResult(x_header, i);
			setResult(x_header, i, x_coor_raw + rs_px_shift);
			y_coor_raw = getResult(y_header, i);
			setResult(y_header, i, y_coor_raw + rs_px_shift);
		}
	} else {
		x_header = header[0];
		y_header = header[1];
		frame_header = header[2];
		intensity_header = header[3];
	}
	if(input_loc_file == "PeakFit"){
		header = sortLocFile(header, frame_header, 100);
		print("Loc-file sorted");
		x_header = header[0];
		y_header = header[1];
		frame_header = header[2];
		intensity_header = header[3];
	}
	
	// Aperture size
	aperture_size = pixelAnalyzerCircleSize(0, 0, radius);
	if(I2_determination == "Fit+Circle"){
		aperture_size2 = pixelAnalyzerCircleSize(0, 0, radius2);
	}
	
	for(i=0; i<nResults; i++){
		showProgress(i+1, nResults);
		
		x_coor_raw = getResult(x_header, i);
		y_coor_raw = getResult(y_header, i);
		x_coor_px = floor(x_coor_raw/pixelsize);
		y_coor_px = floor(y_coor_raw/pixelsize);
		
		if(input_loc_file == "ThunderSTORM" || input_loc_file == "PeakFit") frame_increment = 0;
		else if(input_loc_file == "TRABI_output"){
			if(frame_start_number == 1) frame_increment = 0;
			else frame_increment = 1;
		}
		else frame_increment = 1;
		frame_raw = getResult(frame_header, i);
		frame_ij = getResult(frame_header, i) + frame_increment;
	
		if(free_fit == true){ 
			if(input_loc_file == "rapidSTORM"){
				fwhm_x = getResult(header[17], i);
				fwhm_y = getResult(header[19], i);
			}
			else if(input_loc_file == "ThunderSTORM"){
				sigma_TS = getResult(header[4], i);
			}
			else if(input_loc_file == "PeakFit"){
				sigma_PF_x = getResult(header[5], i);
				sigma_PF_y = getResult(header[6], i);
			}
		}
		old_intensity = getResult(intensity_header, i);
	
		if(frame_ij>nSlices) i = nResults; //more frames in stack than in loc-file (common rapidSTORM error) 
		else { /*** INTENSITY ESTIMATION ***/
	
			// (1) Check if localization is valid
			backward_count = intervalPositionLocFile(i, "backward", frame_header, frame_raw);
			forward_count = intervalPositionLocFile(i, "forward", frame_header, frame_raw);
			neighbor_count = neighboringLocalizations(x_coor_raw, y_coor_raw, x_header, y_header, i, backward_count, forward_count, exclusion_zone_radius_nm); 
			ROI_inside = checkIfRoiInside(x_coor_px, y_coor_px, x_max, y_max, radius);
	
			if(ROI_inside == true && neighbor_count == 1){
				
				// (2) Determine I_raw
				setSlice(frame_ij);
				intensity_array = pixelAnalyzerCircle(x_coor_px, y_coor_px, radius, aperture_size);
				Array.getStatistics(intensity_array, min, max, mean, stdDev);
				raw_intensity = mean*intensity_array.length;
				area = intensity_array.length;
				if(I2_determination == "Fit+Circle"){
					intensity_array2 = pixelAnalyzerCircle(x_coor_px, y_coor_px, radius2, aperture_size2);
					Array.getStatistics(intensity_array2, min, max, mean, stdDev);
					raw_intensity2 = mean*intensity_array2.length;
					area2 = intensity_array2.length;
				}
				// (3) Determine background
				// (3.1) Determine frame information
				start = i + forward_count + 1; 
				base_count = 0;
				averaging = newArray(average_bg_number);
				average_count = 0;
				highlander_count = 0;
				
				for(k=1; k<=nSlices; k++){
					free_space = true;
					free_frame = NaN;
					for(j=start; j<nResults; j++){
						if(getResult(frame_header, j) == frame_raw + k){
							x = getResult(x_header, j);
							y = getResult(y_header, j);
							distance = sqrt(pow((x_coor_raw - x), 2) + pow((y_coor_raw - y), 2));
							if(distance<=exclusion_zone_radius_nm){ 
								free_space = false;
								highlander_count++;
							}
						}				
						else if(getResult(frame_header, j) > frame_raw + k){ 
							start = j; 
							j = nResults; 
						}
						if(exclude_highlander == true && highlander_count > highlander_threshold){
							j = nResults;
						}
					}				
					if(free_space == true) base_count++;
					else if(free_space == false) base_count = 0; 
				
					if(base_count >= basejump && average_count < average_bg_number && frame_raw + k <= nSlices - frame_increment){
						free_frame = frame_raw + k; 
						averaging[average_count] = free_frame;
						average_count++;
					}
	
					if(base_count >= basejump && average_count == average_bg_number && frame_raw + k <= nSlices - frame_increment){
						k = nSlices + 1;
					}
					else if(frame_raw + k > nSlices) k = nSlices + 1;
					if(exclude_highlander == true && highlander_count > highlander_threshold){
							k = nSlices + 1;
					}
				}
				// (3.2) Determine I_BG
				bg_array_average = newArray(average_bg_number);
				for(m=0; m<average_bg_number; m++){
					
					if(averaging[m] == 0) free_frame_ij = NaN;
					else free_frame_ij = averaging[m] + frame_increment;
					
					if(free_frame_ij <= nSlices && free_frame_ij != NaN){
						setSlice(free_frame_ij);
						bg_array = pixelAnalyzerCircle(x_coor_px, y_coor_px, radius, aperture_size);
						Array.getStatistics(bg_array, min, max, mean, stdDev);
						std_surrounding = stdDev;
						bg_intensity = mean*bg_array.length;
						bg_array_average[m] = bg_intensity;
					} else {
						bg_array_average[m] = NaN; 
						std_surrounding = NaN;
					}
				}
				// BG information on free_frames 
				if(background_info == true) free_frame_AV = print2line(averaging, "&");
				else free_frame_AV = NaN;
	
				// (3.3) Analyze BG
				Array.getStatistics(bg_array_average, min, max, mean, stdDev);
				bg_intensity_AV = mean;
	
				// (4) Calculate corrected intensity
				intensity_corrected = raw_intensity - bg_intensity_AV;
				intensity_percentage = old_intensity / intensity_corrected * 100; 
				intensity_difference = intensity_corrected - old_intensity;
				if(I2_determination == "Fit+Circle"){
					intensity_corrected2 = raw_intensity2 - bg_intensity_AV*area2/area;
					intensity_percentage2 = intensity_corrected2 / intensity_corrected * 100;
				}
				// Filtering of P
				if(threshold_P_value == true && (intensity_percentage < lower_P_threshold || intensity_percentage > upper_P_threshold)) intensity_percentage = NaN;
				
				// (5) OUTPUT
				output = defineOutput(output_file_option, input_loc_file, free_fit);
				if(threshold_P_value == true && intensity_percentage >= lower_P_threshold && intensity_percentage<=upper_P_threshold) print2file(TRABI_output_file, output, spacer); 
				else if(threshold_P_value == false) print2file(TRABI_output_file, output, spacer);
				
			} else {
				if(threshold_P_value == false) {
					output = defineOutput("Basic2", input_loc_file, free_fit);
					if(output_file_option != "Basic"){
						no_of_placeholder = header_intensity_file.length - output.length;
						for(d=0; d<no_of_placeholder; d++) output = Array.concat(output, NaN);
					}
					print2file(TRABI_output_file, output, spacer);
				}
			}
		}
	}
	run("Clear Results");
	if (isOpen("Results")) {selectWindow("Results");run("Close");}
	print("Done.", (getTime - time1)/1000 + " s");
	File.close(TRABI_output_file);
	print("");
}
if(batch_processing == true) print("All files processed.");


/******************************* FUNCTIONS *******************************/

function defineOutput(string1, string2, boolean){
	if(string1 == "Basic2"){
		intensity_corrected = NaN; intensity_percentage = NaN;
		if(I2_determination == "Fit+Circle"){intensity_corrected2 = NaN; intensity_percentage2 = NaN;}
	}
	output1 = newArray(x_coor_raw, y_coor_raw, frame_raw, old_intensity, intensity_corrected, intensity_percentage); 
	if(I2_determination == "Fit+Circle"){
		output_r2 = newArray(intensity_corrected2, intensity_percentage2);
		output1 = Array.concat(output1, output_r2);
		}
	if(string2 == "rapidSTORM" && boolean == true) output2 = newArray(fwhm_x, fwhm_y); 
	else if(string2 == "ThunderSTORM" && boolean == true) output2 = sigma_TS;
	else if(string2 == "PeakFit"&& boolean == true) output2 = newArray(sigma_PF_x, sigma_PF_y);
	else output2 = NaN;
	if(string1 != "Basic2") output3 = newArray(intensity_difference, raw_intensity, bg_intensity_AV, free_frame_AV, area, std_surrounding); 
	
	if(string1 == "Basic" || (string1 == "Basic2" && boolean == false)) output = output1;
	else if(string1 == "Basic+FWHM" || (string1 == "Basic2" && boolean == true)) output = Array.concat(output1, output2);
	else if(string1 == "Basic+BG") output = Array.concat(output1, output3);
	else if(string1 == "Complete"){
		output = Array.concat(output1, output2);
		output = Array.concat(output, output3);
	}
	return output;
}

function printHeader(string1, string2, boolean){
	header1 = newArray("#X","Y","Frame", "I_"+string2, "I_1", "P(%)"); 
	if(I2_determination == "Fit+Circle"){
		header1_r2 = newArray("I_2", "P2(%)");
		header1 = Array.concat(header1, header1_r2);
	}
	if(string2 == "rapidSTORM" && boolean == true) header2 = newArray("fwhm_x", "fwhm_y");
	else if(string2 == "ThunderSTORM" && boolean == true) header2 = "sigma";
	else if(string2 == "PeakFit" && boolean == true) header2 = newArray("sigma_x", "sigma_y");
	else header2 = "width";
	header3 = newArray("I_diff", "I_raw,1", "I_BG,1", "Bg-Frame", "ROI-Area", "Bg-StDev");
	
	if(string1 == "Basic") header = header1;
	else if(string1 == "Basic+FWHM") header = Array.concat(header1, header2);
	else if(string1 == "Basic+BG") header = Array.concat(header1, header3);
	else if(string1 == "Complete"){
		header = Array.concat(header1, header2);
		header = Array.concat(header, header3);
	}
	return header;
}

function intervalPositionLocFile(start, direction, label, frame_raw){
	coordinate_count = 0;
	step = 1;
	do {
		if((direction == "backward" && (start-step) >= 0) || (direction == "forward" && (start+step) < nResults)){
			if(direction == "backward") frame_check = getResult(label, start - step);
			else frame_check = getResult(label, start + step);
			step++;
			if(frame_check == frame_raw) coordinate_count++;
		}
		else frame_check = frame_raw + 1;
	} while (frame_check == frame_raw);
	return coordinate_count;
}

function neighboringLocalizations(xc, yc, label_x, label_y, start_increment, start, stop, radius){
	neighbor_count = 0;
	for(j=start_increment-start; j<=start_increment+stop; j++){
		x = getResult(label_x, j);
		y = getResult(label_y, j);
		distance = sqrt(pow((xc-x), 2) + pow((yc-y), 2));
		if(distance<=radius) neighbor_count++; 
	}
	return neighbor_count;
}

function checkIfRoiInside(xc, yc, x_max, y_max, radius){
	inside = true;
	if(xc + radius > x_max-1 || xc - radius < 0 || yc + radius > y_max-1 || yc - radius < 0) inside = false;
	return inside;
}

function pixelAnalyzerCircleSize(xc, yc, radius){
	array = newArray();
	for(y=yc-floor(radius); y<=yc+floor(radius); y++){
		for(x=xc-floor(radius); x<=xc+floor(radius); x++){
			pixel_value = getPixel(x, y);
			distance = sqrt(pow((xc-x), 2) + pow((yc-y), 2));
			if(distance<=radius+radius_increment) array = Array.concat(array, pixel_value); 
		}
	}
	return array.length;
}

function pixelAnalyzerCircle(xc, yc, radius, array_length){
	array = newArray(array_length);
	array_position = 0;
	for(y=yc-floor(radius); y<=yc+floor(radius); y++){
		for(x=xc-floor(radius); x<=xc+floor(radius); x++){
			pixel_value = getPixel(x, y);
			distance = sqrt(pow((xc-x), 2) + pow((yc-y), 2));
			if(distance<=radius+radius_increment){
				array[array_position] = pixel_value;
				array_position++;
			}
		}
	}
	return array;
}

function loadTxtFile(filename, info_line, boolean){
	if(boolean == true){
		array = newArray();
		lineseparator = "\n";
		cellseparator = " ";
		lines = split(File.openAsString(filename), lineseparator);
		for(i=info_line; i<lines.length; i++){
			items = split(lines[i], cellseparator);
			array = Array.concat(array, items[1]);
		}
	} 
	else array = newArray(true, "rapidSTORM", true, true, 100, "Basic", true, 0, 150, false, false, 7, 2, 7, "Fit_only", 2, true, 100, 2, ".tif", ".txt");
	return array;
}

function loadLocFile(filename, header, info_line, cellseparator){
	lineseparator = "\n";
	lines = split(File.openAsString(filename), lineseparator);
	if(input_loc_file == "rapidSTORM"){
		labels = wordSplitter(lines[0], "field identifier");
		labels = redefineLabels(labels, header); 
	}
	else labels = split(lines[0], cellseparator);

	items_position = newArray();
	for(j=0; j<labels.length; j++){
		header_wanted = false;
		for(k=0; k<header.length; k++){
			if(labels[j] == header[k]){
				header_wanted = true; 
				items_position = Array.concat(items_position, j);
			}
		}
		if(header_wanted == true) setResult(labels[j],0,0);
	}
	run("Clear Results");
	
	for(i=info_line; i<lines.length; i++){
	items = split(lines[i], cellseparator);
		for(j=0; j<items_position.length; j++){
			position = items_position[j];
			setResult(labels[position], i-info_line, parseFloat(items[position]));
		}
	}
}

function wordSplitter (string1, delimiter){
	string2 = replace(string1, delimiter, "@");
	parts = split(string2, "@");
	for(i=0; i<parts.length; i++)
	return parts;
}

function redefineLabels(labels, regex_array){
	if(matches(labels[0], ".*insequence.*")){
		labels = Array.slice(labels, 1, labels.length);
	}
	count_x = 0; count_y = 0; count_ADC = 0;
	for(v=0; v<header.length-1; v+=2){
		for(w=0; w<labels.length; w++){
			if(matches(labels[w], header[v])){
				labels[w] = header[v+1];
				if(labels[w] == header[1]) count_x++; 
				if(labels[w] == header[1] && count_x > 1) labels[w] = header[1] + count_x;
				if(labels[w] == header[3]) count_y++; 
				if(labels[w] == header[3] && count_y > 1) labels[w] = header[3]+count_y;
				if(labels[w] == header[7]) count_ADC++; 
				if(labels[w] == header[7] && count_ADC > 1) labels[w] = header[7]+count_ADC;
			}
		}
	}
	return labels;
}

function suffixEliminator(old_name, file_ending){
	length_of_new_name = indexOf(old_name, file_ending);
	new_name = substring(old_name, 0, length_of_new_name); 

	return new_name;
}

function filenameDistillery(filename, directory){
	path_name = File.getName(directory);
	path_length = lengthOf(path_name);
	pos_string = indexOf(filename, path_name);
	pos_string2 = pos_string + path_length + 1;
	filename_length = lengthOf(filename);
	new_string = substring(filename, pos_string2, filename_length);
	
	return new_string;
}

function print2file(file, output, token){
	line = "";
	for (v=0; v<output.length; v++) {
		if(v == output.length-1) token = "";
		line = line + output[v] + token; 
	}
	print(file, line);
}

function print2line(output, token){
	line = "";
	for (v=0; v<output.length; v++) {
		if(v == output.length-1) token = "";
		line = line + output[v] + token;
	}
	return line;
}

function sortLocFile(header, frame_header, frame_spread){ 
	//sorting for PeakFit
	RT_size = nResults;
	run("Summarize");
	frame_min = getResult("Frame", RT_size+2);
	frame_max = getResult("Frame", RT_size+3);
	newRT_line = 0;
	setResult("srt_pos", 0, 0); 
	sorting_index = 1; 
	start_position = 0;
	counter = 0; 
	stop_position = round(RT_size / frame_max * frame_spread); // frame spread: depends on deviation of frame numbers within RT (factor determines speed)
	
	for(j=frame_min; j<=frame_max; j++){
		showProgress(j+1, frame_max);
		sequence_interrupted = false;
		for(i=start_position; i<start_position+stop_position && i<RT_size; i++){
			counter = start_position;
			if(getResult("srt_pos", i) == sorting_index && sequence_interrupted == false) counter++;
			else sequence_interrupted = true;
			line_array = newArray(header.length);
			frame = getResult(frame_header, i);
			if(frame == j){
				for(k=0; k<header.length; k++) line_array[k] = getResult(header[k], i); 
				for(k=0; k<header.length; k++) setResult(header[k]+"_srt", newRT_line, line_array[k]);
				setResult("srt_pos", i, sorting_index);
				newRT_line++;
			}
			start_position = counter; 
		}
	}
	IJ.deleteRows(RT_size, RT_size+3);
	updateResults();
	for(k=0; k<header.length; k++) header[k] = header[k]+"_srt";
	return header;
}

function html_message(){
text = "<html>"
	+"<h3>TRABI</h3>"
	+"Temporal, Radial-Aperture Based Intensity-estimation<br>"
	+"<h4>BATCH PROCESSING</h4>"
	+"When selected, all files to be processed must be placed into a single folder while image stack and corresponding localization file "
	+"should be named in the same way, e.g. 'dSTORM.tif', 'dSTORM.txt', 'dSTORM_1.tif', 'dSTORM_1.txt', 'sal1xalB_a.tif', 'sal1xalB_a.txt' etc. " 
	+"A separate dialog window will appear, where <b>Image file extension</b> (typically .tif) and <b>Loc-file extension</b> (typically .txt or .csv) have to be defined. "
	+"TRABI files will be saved in a separate folder.<br>" 
	+"<h4>I. Input options</h4>"
	+"<b>Loc-file:</b> select type of localization file, e.g., rapidSTORM localization file.<br>" 
	+"<b>Free PSF fit:</b> use when free fitting was performed and output of this information is desired (needs output file 'Basic+FWHM' or 'Complete').<br>"
	+"<b>Coordinates in nm:</b> activate if coordinates in the SMLM localization file are in nm, too.<br>"
	+"<b>Pixelsize:</b> use exactly the same pixel size as in the SMLM software.<br>"
	+"<h4>II. Output options</h4>"
	+"<b>Output file:</b> determine the complexity of the output file with 'Basic' being the most simple.<br>"
	+"<b>P-filter activated:</b> activate if <I>P</I> should be confined in the output file using P-min and P-max as lower and upper thresholds, respectively; " 
	+ "<I>P</I> = <I>I_Fit</I>/<I>I_1</I>.<br>"
	+"<b>BG-Averaging: save frame info:</b> activate if the temporal information of the background estimation, i.e., the frame numbers of the individual background frames, "
	+ "should be saved (only for output options 'Basic+BG' and 'Complete').<br>"
	+"<b>Save settings to file name:</b> activate if the file name should contain settings information.<br>"
	+"<h4>III. TRABI settings</h4>"
	+"<b>r1:</b> define radius <I>r_1</I> for intensity estimation of <I>I_1</I>.<br>"
	+"<b>Basejump:</b> use to skip #<I>basejump-1</I> frames directly after the localization itself or an interference. A value of 1 deactivates it.<br>" 
	+"<b>nBG:</b> number of frames used for background averaging. <br>"
	+"<b>I2 by:</b> define how <I>I_2</I> should be determined, i.e., by fitting only or additionally by a central aperture. <br>" 
	+ "Selection of 'Fit+Circle' leads to a second dialog where <I>r_2</I> must be defined. The output is then supplemented by additional values, i.e., <I>I_2</I> and <I>P_2</I> = <I>I_2</I>/<I>I_1</I>.<br>"
	+"<b>EZ factor:</b> define exclusion zone (EZ) factor. The radius of the EZ aperture is defined by EZ_factor*<I>r_1</I>. For accurate intensity estimation we strongly recommend the default value of 2.0. " 
	+ "If the amount of localization matters, the factor might be reduced. <br>" 
	+"<b>Highlander filter:</b> activate to reject spots, which reside permanently in the fluorescent on-state by using a threshold (in frames); the threshold is the amount of background frames "
	+" that are allowed to be skipped before the localization is rejected.<br>"
	+"<br>"
	+"<small><A HREF=dx.doi.org/10.1038/nmeth.4073#*>Paper</A>: Christian Franke, Markus Sauer, Sebastian van de Linde. <I>Nat. Methods</I>, <B>14</B>, 41-44 (2017)</small><br>"
	+"</font>";
	return text;
}
