/* LocFileVisualizer
 *  
 *  Localization file visualizer
 *
 *  Import and visualization of single-molecule localizations. 
 *  Supports 2D histograms, 3D image stacks, quantitative localization images, 
 *  visualization of localizations within raw TIF stacks.
 *  User-defined import options are supported as well.
 *
 *  2014-2019
 *  
 *  Author: Sebastian van de Linde, s.vandelinde@strath.ac.uk
 *  University of Wuerzburg, Germany (until 2016)
 *  University of Strathclyde, Glasgow, Scotland
 */


/***** GUI *****/
version = "1.1";
presets_file = getDirectory("macros") + "LocFileVisualizer_v1.1_settings.txt";
presets_exist = File.exists(presets_file);
presets = loadTxtFile(presets_file, 0, presets_exist);
if(presets.length != 28) presets = loadTxtFile(presets_file, 0, false);
items0 = newArray("Import_loc_file", "Generate_image", "Import&Generate", "Show_localizations", "Import&Show");
items1 = newArray("rapidSTORM", "ThunderSTORM", "Basic", "User-defined");
items2 = newArray("8-bit", "16-bit", "32-bit");
items3 = newArray("All", "Odd", "Even");
items4 = newArray("Linear", "None");
html = html_message();
Dialog.create("LocFileVisualizer " + version);
Dialog.setInsets(0,0,0);
Dialog.addMessage("I. Import options");
Dialog.addChoice("Operation", items0, presets[0]);
Dialog.addChoice("Loc-file", items1, presets[1]);
Dialog.setInsets(0,0,0);
Dialog.addMessage("II. Options for image generation");
Dialog.addNumber("Pixel size XY", presets[2], 0, 3, "nm");
Dialog.addNumber("Pixel size Z", presets[3], 0, 3, "nm");
Dialog.addNumber("Frame start", presets[4], 0, 6, "");
Dialog.addNumber("Frame end", presets[5], 0, 6, "");
Dialog.addNumber("Intensity min", presets[6], 0, 6, "");
Dialog.addNumber("Intensity max", presets[7], 0, 6, "");
Dialog.addChoice("Image bit depth", items2, presets[8]);
Dialog.addChoice("Frames used", items3, presets[9]);
Dialog.addCheckbox("z-stack", presets[10]);
Dialog.addCheckbox("Gray value quantification", presets[11]);
Dialog.addCheckbox("Bilin. pixel interpolation (32-bit)", presets[12]);
Dialog.addCheckbox("Image corner set to (0,0)", presets[13]);
Dialog.addHelp(html);
Dialog.show();
operation = Dialog.getChoice();
input_loc_file = Dialog.getChoice();
pixelsize = Dialog.getNumber(); 
z_depth = Dialog.getNumber(); 
frame_start = Dialog.getNumber();
frame_end = Dialog.getNumber();
Intensity_min = Dialog.getNumber(); 
Intensity_max = Dialog.getNumber();
bitdepth = Dialog.getChoice();
split_frames = Dialog.getChoice();
z_stack = Dialog.getCheckbox(); 
quantification = Dialog.getCheckbox();
pixel_interpolation = Dialog.getCheckbox();
set_imagecorner_to_zero_coordinates = Dialog.getCheckbox();

//new presets
new_presets = newArray(operation,input_loc_file,pixelsize,z_depth,frame_start,frame_end,Intensity_min,Intensity_max,bitdepth,split_frames,z_stack,quantification,pixel_interpolation,set_imagecorner_to_zero_coordinates);

//RT v1.1
if(operation == "Generate_image" || operation == "Show_localizations"){
	if(isOpen("Results") == false) exit("No localization file loaded");
}

//GUI User defined header
if(input_loc_file == "User-defined" && operation != "Generate_image" && operation != "Show_localizations"){ // v1.1
	items5 = newArray("Space", "Tab", "Comma", "Semicolon");
	items5a = newArray(" ", "\t", ",", ";");
	Dialog.create("LocFileVisualizer: User-defined loc file");
	Dialog.addMessage("Specifiy columns: blank fields will not be imported");
	Dialog.addMessage("Note: values start with 0!");
	Dialog.addNumber("X coordinate", presets[14], 0, 2, "column");
	Dialog.addNumber("Y coordinate", presets[15], 0, 2, "column");
	Dialog.addNumber("Z coordinate", presets[16], 0, 2, "column");
	Dialog.addNumber("Frame", presets[17], 0, 2, "column");
	Dialog.addNumber("Intensity", presets[18], 0, 2, "column");
	Dialog.addNumber("X spot width", presets[19], 0, 2, "column");
	Dialog.addNumber("Y spot width", presets[20], 0, 2, "column");
	Dialog.addNumber("First line in file with values", presets[21], 0, 2, "line (0: usually header)");
	Dialog.addChoice("Separator", items5, presets[22]);
	Dialog.show;
	user_x = Dialog.getNumber();
	user_y = Dialog.getNumber();
	user_z = Dialog.getNumber();
	user_frame = Dialog.getNumber();
	user_int = Dialog.getNumber();
	user_width_x = Dialog.getNumber();
	user_width_y = Dialog.getNumber();
	user_first_line_with_coordinates = Dialog.getNumber();
	user_separator = Dialog.getChoice();
	user_array = newArray(user_x, user_y, user_z, user_frame, user_int, user_width_x, user_width_y);
	user_array1 = newArray();
	user_array2 = newArray("#X", "Y", "Z", "Frame", "Intensity", "Width_X", "Width_Y");
	user_header = newArray();
	for(i=0; i<user_array.length; i++){
		if(isNaN(user_array[i]) == false){
			user_header = Array.concat(user_header, user_array2[i]);
			user_array1 = Array.concat(user_array1, user_array[i]);
		}
	}
	pos = positionInArray(user_separator, items5);
	user_separator = items5a[pos];
	
	//new presets
	user_separator_new_presets = items5[pos];
	new_presets_User = newArray(user_x,user_y,user_z,user_frame,user_int,user_width_x,user_width_y,user_first_line_with_coordinates,user_separator_new_presets);
} else {
	new_presets_User = Array.slice(presets, 14, presets.length-5);
	user_first_line_with_coordinates = 1; // v1.1
	user_separator = " "; // v1.1
}
//GUI Show localizations
if(operation == "Show_localizations" || operation == "Import&Show"){
	if(nImages == 0){
		 print("");
		 path = File.openDialog("Please, open a TIF-stack"); 
		 open(path);
		 stack_name = File.getName(path);
		 print("Stack loaded:", stack_name);
	}
	items6 = newArray("Point", "Circle", "Box");
	items7 = newArray("Yellow", "Green", "Blue", "Orange", "Red", "Magenta",  "Cyan", "Black", "White");
	items8 = newArray("0", "1");
	Dialog.create("LocFileVisualizer: Show localizations");
	Dialog.addMessage("Specifiy visualization");
	Dialog.addChoice("Type", items6, presets[23]);
	Dialog.addChoice("Colour", items7, presets[24]);
	Dialog.addNumber("Radius of selection", presets[25], 0, 3, ".5 px");
	Dialog.addMessage("Specifiy camera pixel size");
	Dialog.addNumber("Pixel size", presets[26], 0, 3, "nm");
	Dialog.addMessage("Specifiy how FRAMES are counted");
	Dialog.addChoice("Starting from", items8, presets[27]);
	Dialog.show;
	selection_type = Dialog.getChoice();
	selection_colour = Dialog.getChoice();
	selection_size = Dialog.getNumber();
	camera_pixelsize = Dialog.getNumber();
	frame_counter = Dialog.getChoice();	
	if(input_loc_file == "rapidSTORM"){
		pixel_shift = 0.5 * camera_pixelsize;
	} else pixel_shift = 0;
	
	//new presets
	new_presets_ShowLoc = newArray(selection_type,selection_colour,selection_size,camera_pixelsize,frame_counter);
} else {
	new_presets_ShowLoc = Array.slice(presets, presets.length-5);
}

// Save new presets
if(presets_exist == true){ //if file is in Fiji macro folder
	new_presets = Array.concat(new_presets,new_presets_User);
	new_presets = Array.concat(new_presets,new_presets_ShowLoc);
	f = File.open(presets_file);
	presets_description = newArray("Operation","LocFile","PixelsizeXY","PixelsizeZ","Frame_start","Frame_end","Intensity_min","Intensity_max","Bit_depth","Frames_used","zStack","Quantification","Interpolation","ImageCornerZero","User_X","User_Y","User_Z","User_Frame","User_Intensity","X_width","Y_width","Values_first_line","Separator","Selection_type","Selection_colour","Selection_size","Camera_pixelsize","Frame_counter");
	for(i=0; i<presets_description.length; i++){
		line = presets_description[i] + " " + new_presets[i];
		print(f, line);
	}
	File.close(f);
}

if(isNaN(frame_start) == true) frame_start = 0;
if(isNaN(frame_end) == true) frame_end = 1E99;
if(isNaN(Intensity_min) == true) Intensity_min = 0;
if(isNaN(Intensity_max) == true) Intensity_max = 1E99; 

setBatchMode(true);
print("");

/***** PART I: Load loc file *****/
// HEADER
if(input_loc_file == "rapidSTORM") header = newArray(".*Position-0-0.*", "x", ".*Position-1-0.*", "y", ".*ImageNumber-0-0.*", "frame", ".*Amplitude-0-0.*", "ADC", ".*TwoKernelImprovement-0-0.*", "TwoKernel", ".*FitResidues-0-0.*", "FitResidues", ".*LocalBackground-0-0.*", "LocalBackground", ".*Position-2-0.*", "z", ".*PSFWidth-0-0.*", "PSFwidthFWHMx", ".*PSFWidth-1-0.*", "PSFwidthFWHMy");
else if(input_loc_file == "ThunderSTORM") header = newArray("\"x [nm]\"","\"y [nm]\"", "\"frame\"","\"intensity [photon]\"", "z [nm]", "\"sigma [nm]\"","\"offset [photon]\"","\"bkgstd [photon]\"","\"chi2\"","\"uncertainty [nm]\"");
else if(input_loc_file == "Basic") header = newArray("#X", "Y", "Frame", "Intensity", "Z");
else if(input_loc_file == "User-defined") header = newArray("#X", "Y", "Frame", "Intensity", "Z", "Width_X", "Width_Y");

if(input_loc_file == "User-defined") first_line_with_coordinates = user_first_line_with_coordinates;
else first_line_with_coordinates = 1;

if(input_loc_file == "ThunderSTORM") separator = ",";
else if(input_loc_file == "User-defined"){
	separator = user_separator;
}
else separator = " ";

run("Set Measurements...", "area center integrated redirect=None decimal=2");
run("Conversions...", " "); // no scaling when transfered from 32 to 16 bit etc.

if (isOpen("Results") && (operation == "Import_loc_file" || operation == "Import&Generate")) {
    selectWindow("Results");
    run("Close");
}

if(operation == "Import_loc_file" || operation == "Import&Generate" || operation == "Import&Show"){
	file = File.openDialog("Select rapidSTORM loc file");
	filename = File.getName(file);
	print("Loading " + filename);
	time_start = getTime;
	if(input_loc_file == "User-defined") header_for_extraction = loadLocFileUser(file, user_array1, user_header, first_line_with_coordinates, separator);
	else header_for_extraction = loadLocFile(file, input_loc_file, header, first_line_with_coordinates, separator);
	updateResults();
}

/**** Define Header ****/
if(operation != "Import_loc_file"){
	run("Summarize");
	results_length = nResults;
	results_length = results_length-4;
	
	if(input_loc_file == "rapidSTORM"){
		x_header = header[1];
		y_header = header[3];
		frame_header = header[5];
		intensity_header = header[7];
		z_header = header[15];
	}
	else if(input_loc_file == "User-defined"){
		x_header = header[0];
		y_header = header[1];
		frame_header = header[2];
		intensity_header = header[3];
		z_header = header[4];
	}
	else {
		x_header = header[0];
		y_header = header[1];
		frame_header = header[2];
		intensity_header = header[3];
		z_header = header[4];
	}
}

/**** PART II: Show Localizations ****/
if(operation == "Show_localizations" || operation == "Import&Show"){
	time_start2 = getTime;
	run("Point Tool...", "type=Crosshair color="+selection_colour+" size=Large");
	run("Properties...", "pixel_width=1 pixel_height=1 voxel_depth=1");
	
	if (isOpen("ROI Manager")){
		selectWindow("ROI Manager");
		run("Close");
	}
	roiManager("Associate", "true");
	nlocs = results_length; 
	selection_size = (selection_size+0.5)*2; 
	for(i=0; i<nlocs; i++){
		x = floor((getResult(x_header, i) + pixel_shift)/camera_pixelsize); 
		y = floor((getResult(y_header, i) + pixel_shift)/camera_pixelsize);
		if(input_loc_file == "ThunderSTORM" || input_loc_file == "PeakFit") frame = getResult(frame_header, i); 
		else frame = getResult(frame_header, i) + 1;
		if(frame_counter == 1) frame = getResult(frame_header, i);
		else frame = getResult(frame_header, i) + 1;
		
		if(frame>nSlices) i = nResults;
		else {
			setSlice(frame);
			labelLocalization(x, y, selection_type, selection_size);
			roiManager("Add");
		}
	}
	roiManager("Show All");
	print(nlocs + " localizations");
}

/**** PART III: Make Histogram ****/
if(operation == "Generate_image" || operation == "Import&Generate"){
	time_start3 = getTime;

	if(isNaN(getResult(z_header, 0)) == true && z_stack == true) exit("z-stack cannot be generated as no z-info is provided");

	// MIN & MAX VALUES
	x_min = getResult(x_header, results_length+2);
	x_max = getResult(x_header, results_length+3);
	y_min = getResult(y_header, results_length+2);
	y_max = getResult(y_header, results_length+3);
	if(z_stack == true){
		z_min = getResult(z_header, results_length+2);
		z_max = getResult(z_header, results_length+3);
	}
	
	// X-BINNING
	if(set_imagecorner_to_zero_coordinates == false) start_x = floor(x_min / pixelsize); 
	else start_x = 0;
	end_x = floor(x_max / pixelsize);
	delta_end_x = x_max % pixelsize;
	if(delta_end_x > 0) end_x++;
	x_range = end_x - start_x;
	
	// Y-BINNING
	if(set_imagecorner_to_zero_coordinates == false) start_y = floor(y_min / pixelsize);
	else start_y = 0;
	end_y = floor(y_max / pixelsize);
	delta_end_y = y_max % pixelsize;
	if(delta_end_y > 0) end_y++;
	y_range = end_y - start_y;
	
	// Z-BINNING
	if(z_stack == true){
		start_z = floor(z_min / z_depth);
		end_z = floor(z_max / z_depth);
		delta_end_z = z_max % z_depth;
		if(delta_end_z > 0) end_z++;
		z_range = end_z - start_z + 1;
	} else z_range = 1;
	
	if(operation == "Generate_image") filename = "SMLM" + "_XY_"+pixelsize;
	if(split_frames != "All") filename = filename + "_" + split_frames;
	newImage(filename, bitdepth+" black", x_range, y_range, z_range);
	
	print("Generating image");
	for(i=0; i<results_length; i++){
		showProgress(i+1, results_length);
		x_coor = getResult(x_header, i);
		y_coor = getResult(y_header, i);
	
		if(z_stack == true) z_coor = getResult(z_header, i);
	
		x_raw = x_coor/pixelsize;
		y_raw = y_coor/pixelsize;
	
		x_px = floor(x_raw) - start_x;
		y_px = floor(y_raw) - start_y;
		xc = floor(x_raw) + 0.5; 
		yc = floor(y_raw) + 0.5;
	
		if(z_stack == true) z = floor(z_coor/z_depth) - start_z;
	
		frame = getResult(frame_header, i);
		spot_intensity = getResult(intensity_header, i);
	
		selectWindow(filename);
		if(z_stack == true) setSlice(z+1);
		if(quantification == true) k = 1;
		else k = spot_intensity;
	
		if(pixel_interpolation == true){
			dx = xc - x_raw;
			if(dx < 0) x_dir = 1;
			else if(dx > 0) x_dir = -1;
			else x_dir = 0;

			dy = yc - y_raw;
			if(dy < 0) y_dir = 1;
			else if(dy > 0) y_dir = -1;
			else y_dir = 0;
	
			dx_a = abs(dx);
			dy_a = abs(dy);

			source_pixel_perc = (1 - dx_a) * (1 - dy_a);
			x_neigh_perc = dx_a * (1 - dy_a);
			y_neigh_perc = (1 - dx_a) * dy_a;
			xy_neigh_perc = dx_a * dy_a;
		}
	
		if(split_frames != "All"){
			if(split_frames == "Odd") AT = 0; 
			else if(split_frames == "Even") AT = 1;
			framemod = abs(frame%2-AT);
		} else framemod = 0;
		
		// "Expression filter"
		if(framemod == 0 && frame >= frame_start && frame <= frame_end && spot_intensity >= Intensity_min && spot_intensity <= Intensity_max){/// && two_kernel <= two_kernel_max
	
			//Interpolation
			if(pixel_interpolation == true){
				pixelInterpolation(x_px, y_px, 0, 0, source_pixel_perc, k);
				pixelInterpolation(x_px, y_px, 0, y_dir, y_neigh_perc, k);
				pixelInterpolation(x_px, y_px, x_dir, 0, x_neigh_perc, k);
				pixelInterpolation(x_px, y_px, x_dir, y_dir, xy_neigh_perc, k);
			}
			//standard binning
			else {
				value = getPixel(x_px, y_px);
				value = getPixel(x_px, y_px);
				value = value + 1 * k;
				setPixel(x_px, y_px, value);
			}	
		}
	}
	run("Enhance Contrast", "saturated=0.35");
	if(operation == "Import&Generate") setMetadata("Info", header_for_extraction);
	
	if(z_stack == true) run("Properties...", "channels=1 slices=z_range frames=1 unit=nm pixel_width=pixelsize pixel_height=pixelsize voxel_depth=z_depth frame=[0 sec] origin=0,0");
}
setBatchMode(false);
time_end = getTime;
if(operation == "Show_localizations") time_diff = time_end-time_start2;
else if(operation == "Generate_image") time_diff = time_end-time_start3;
else time_diff = time_end-time_start;
print("Total processing time (s): "+(time_diff)/1000);


/**** Functions ****/

function pixelInterpolation(x, y, x_dir, y_dir, percentage, intensity){
	value = getPixel(x + x_dir, y + y_dir);
	value = value + percentage * intensity;
	setPixel(x + x_dir, y + y_dir, value);
	updateDisplay();
}

function loadLocFile(filename, input_loc_file, header, info_line, cellseparator){
	lineseparator = "\n";
	lines = split(File.openAsString(filename), lineseparator);
	header_for_extraction = lines[0];
	if(input_loc_file == "rapidSTORM"){
		labels = wordSplitter(lines[0], "field identifier");
		labels = redefineLabels(labels, header); 
	}
	else labels = split(lines[0], cellseparator);

	items_position = newArray();
	for(j=0; j<labels.length; j++){
		header_wanted = false;
		for(k=0; k<header.length; k++){
			if(labels[j] == header[k]){
				header_wanted = true; 
				items_position = Array.concat(items_position, j);
			}
		}
		if(header_wanted == true) setResult(labels[j],0,0);
	}
	run("Clear Results");
	
	for(i=info_line; i<lines.length; i++){
		items = split(lines[i], cellseparator);
		for(j=0; j<items_position.length; j++){
			position = items_position[j];
			setResult(labels[position], i-info_line, parseFloat(items[position]));
		}
	}
	return header_for_extraction;
}

function loadLocFileUser(filename, user_defined, header, info_line, cellseparator){
	lineseparator = "\n";
	lines = split(File.openAsString(filename), lineseparator);
	header_for_extraction = print2line(header, " ");

	for(j=0; j<header.length; j++){
		 setResult(header[j],0,0);
	}
	run("Clear Results");
	
	for(i=info_line; i<lines.length; i++){
		items = split(lines[i], cellseparator);
		for(j=0; j<user_defined.length; j++){
			position = user_defined[j];
			setResult(header[j], i-info_line, parseFloat(items[position]));
		}
	}
	return header_for_extraction;
}

//rapidSTORM
function fixCorruptedLine(array1, array2, i) {
	diff = array1.length - array2.length;
	diff_array = newArray(diff);
	for(z = 0; z < diff; z++) diff_array[z] = "NaN";
	array2 = Array.concat(array2, diff_array);
	return array2;
}

//rapidSTORM
function wordSplitter (string1, delimiter){
	string2 = replace(string1, delimiter, "@");
	parts = split(string2, "@");
	for(i=0; i<parts.length; i++)
	return parts;
}

//rapidSTORM
function redefineLabels(labels, regex_array){
	if(matches(labels[0], ".*insequence.*")){
		labels = Array.slice(labels, 1, labels.length);
	}
	count_x = 0; count_y = 0; count_ADC = 0;
	for(v=0; v<header.length-1; v+=2){
		for(w=0; w<labels.length; w++){
			if(matches(labels[w], header[v])){
				labels[w] = header[v+1];
				if(labels[w] == header[1]) count_x++; 
				if(labels[w] == header[1] && count_x > 1) labels[w] = header[1] + count_x;
				if(labels[w] == header[3]) count_y++; 
				if(labels[w] == header[3] && count_y > 1) labels[w] = header[3]+count_y;
				if(labels[w] == header[7]) count_ADC++; 
				if(labels[w] == header[7] && count_ADC > 1) labels[w] = header[7]+count_ADC;
			}
		}
	}
	return labels;
}

function labelLocalization(x, y, type, size){
	if(type == "Circle") makeOval(x-floor(size/2), y-floor(size/2), size, size);
	else if(type == "Box") makeRectangle(x-floor(size/2), y-floor(size/2), size, size);
	else if(type == "Point") makePoint(x, y);
}

function print2line(output, token){
	line = "";
	for (v=0; v<output.length; v++) {
		if(v == output.length-1) token = "";
		line = line + output[v] + token;
	}
	return line;
}

function positionInArray(input, array){
	for(i=0; i<array.length; i++){
		if(array[i] == input){
			position = i;
			i = array.length;
		}
	}
	return position;
}

function loadTxtFile(filename, info_line, boolean){
	if(boolean == true){
		array = newArray();
		lineseparator = "\n";
		cellseparator = " ";
		lines = split(File.openAsString(filename), lineseparator);
		for(i=info_line; i<lines.length; i++){
			items = split(lines[i], cellseparator);
			array = Array.concat(array, items[1]);
		}
	} 
	else array = newArray("Import_loc_file","rapidSTORM",20,20,0,NaN,0,NaN,"32-bit","All",false,true,false,false,0,1,2,3,4,NaN,NaN,1,"Space","Circle","Yellow",3,100,1);
	return array;
}

function html_message(){
text = "<html>"
	+"<h3>LocFileVisualizer</h3>"
	+"Import and visualization of single-molecule localizations.<br>"
	+"Last GUI settings are automatically saved if settings file is provided in Fiji's macro folder.<br>"
	+"<h4>Import options</h4>"
	+"<b>Operation:</b> select one of the following: i) <i>Import_loc_file</i>: localization file is loaded as ImageJ Results table, ii) <i>Generate_image</i>: image generated from Results table, "
	+"iii) <i>Import&Generate</i>: execute steps i) & ii), iv) <i>Show_localizations</i>: Displays localization within TIF stack, or v) <i>Import&Show</i>: execute steps i) & iv).<br>"
	+"In case of iv) or v) another GUI will appear where different specifications are required, i.e. type, colour and size of the selection, camera pixel size and how the "
	+" localization software counts frames (starting either from 0 or 1).<br>"
	+"<b>Loc-file:</b> select type of localization file, e.g. <i>rapidSTORM</i> or <i>ThunderSTORM</i> localization file. <i>Basic</i> supports a localization file format with X, Y, Z, Frame and Intensity information "
	+"and space as separator. If <i>user-defined</i> is selected another GUI will appear where different fields such as the separator (space, tab, comma, semicolon) "
	+"and the first line with coordinates must be specified.<br>" 
	+"It is considered that all coordinates in the localization file are provided in nanometer. If pixel coordinates are used, adjust values for the pixel size.<br>"
	+"<h4>Options for image generation</h4>"
	+"<b>Pixel size XY:</b> specify lateral pixel size that is used for image generation.<br>"
	+"<b>Pixel size Z:</b> specify axial pixel size that is used for generating an axial image stack.<br>"
	+"<b>Frame start:</b> specify first frame that is used for image generation; 0 is 1st frame.<br>"
	+"<b>Frame end:</b> specify last frame that should be used for image generation. If left blank the maximum frame in the list is used<br>"
	+"<b>Intensity min:</b> Minimum intensity threshold.<br>"
	+"<b>Intensity max:</b> Maximum intensity threshold. If left blank no upper threshold is applied.<br>"
	+"<b>Image bit depth:</b> Select between 8, 16 and 32 bit. 32 bit is needed for bilinear interpolation and decimal pixel values.<br>"
	+"<b>Frames used:</b> specify which frames are used for image generation, i.e. all, odd or even. Helpful for FRC analysis (needs a set of two images, e.g. odd and even) <br>"
	+"<br>"
	+"<b>z-stack:</b> If activated a z-stack is generated; stack size depends on min and max z-values as well as 'pixel size Z'.<br>"
	+"<b>Gray value quantification:</b> If activated only the localization count is used for image generation. If deactivated, localization count and spot intensity are used.<br>"
	+"<b>Bilin. pixel interpolation (32-bit):</b> If activated, the intensity of a localization is distributed over neighbouring pixels (max. 4 pixels); "
	+ "works only for 32-bit images<br>"
	+"<b>Image corner set to (0,0):</b> If activated the image starts at XY position (0,0). Otherwise the min coordinate values in the localization file are used.<br>"
	+"<br>"
	+"<small> Sebastian van de Linde, <I>J. Phys. D: Appl. Phys.</I> <B>52</B>, 203002  (<A HREF=https://doi.org/10.1088/1361-6463/ab092f>2019</A>), <A HREF=http://bcp.phys.strath.ac.uk/photophysics/super-resolution/software/>updates</A></small><br>"
	+"</font>";
	return text;
}